% "Survey density forecast comparison in small samples" 
% by Laura Coroneo, Fabrizio Iacone and Fabio Profumo, 
% International Journal of Forecasting, 2024

% This code replicates the empirical exercise and results of Tables 3-18

clear; clc;

%% specification

loss_type = 1;  % 1 for RPS (main results); 2 for QPS (appendix D)

% declare variables
sample_sel = [datetime('01-01-2000','InputFormat','MM-dd-yyyy') datetime('12-01-2019','InputFormat','MM-dd-yyyy'); ...
    datetime('01-01-2000','InputFormat','MM-dd-yyyy') datetime('12-01-2009','InputFormat','MM-dd-yyyy'); ...
    datetime('01-01-2010','InputFormat','MM-dd-yyyy') datetime('12-01-2019','InputFormat','MM-dd-yyyy')];

variable = {'UN'; 'GDP'; 'HICP'}; 

bench_label = {'Uni'; 'UG'; 'GRW'; 'GRWD'; 'Naive'};

SPF = cell(size(sample_sel,1),length(variable));
real = cell(size(sample_sel,1),length(variable));

for i = 1 : size(sample_sel,1)

    for j = 1 : length(variable)

        %% 1. Load data
        [SPF{i,j}, real{i,j}, hist{i,j}] = load_data(variable{j}, sample_sel(i,:));

        %% 2. Compute benchmarks (T,B,H,n_bench)
        bench{i,j} = benchmarks(variable{j}, SPF{i,j}, hist{i,j}, bench_label);

        %% 3. Compute loss (T x H)
        [SPF_loss{i,j},SPF_e{i,j}] = loss(squeeze(real{i,j}.realised), SPF{i,j}.data, SPF{i,j}.edges, loss_type);
        
        % collect average losses
        tab_6(:,1,i,j) = mean(SPF_loss{i,j})';
        
        for b = 1 : size(bench_label,1)

            [b_loss{i,j,b}, b_e{i,j,b}] = loss(squeeze(real{i,j}.realised), bench{i,j}(:,:,:,b), SPF{i,j}.edges, loss_type);
            tab_6(:,b+1,i,j) = mean(b_loss{i,j,b})';

            % compute differential d_t{i,j,b,type}; type = 1 for EPA, type = 2 for FE: 
            
            % EPA
            d_t{i,j,b,1} =  b_loss{i,j,b} - SPF_loss{i,j};
           
            for h = 1 : 2
                for t = 1:size(SPF{i,j}.data,1)
                    ind = ~isnan(SPF_e{i,j}(t,:,h));
                    % FE
                    temp = (SPF_e{i,j}(t,ind,h)-b_e{i,j,b}(t,ind,h));
                    d_t{i,j,b,2}(t,h) =  SPF_e{i,j}(t,ind,h)*temp';
                    denom(t,h) = (temp*temp');
                end

                % forecast combination weights
                lambda(i,j,b,h) = mean(d_t{i,j,b,2}(:,h))/mean(denom(:,h));
                clear denom;

                % collect summary stats for d_t
                for d = 1 : 2
                    temp = autocorr(d_t{i,j,b,d}(:,h));
                    tab_7(:,h,b,i,j,d)= [mean(d_t{i,j,b,d}(:,h)); std(d_t{i,j,b,d}(:,h)); temp([1 2 3 4 8]+1)];
                
                    %% DM test
                    [DM_WCE(h,b,i,j,d), reject_fix_b(h,b,i,j,d),~, ...
                        DM_WPE(h,b,i,j,d), reject_fix_m(h,b,i,j,d),~] = dm(d_t{i,j,b,d}(:,h));
                end
            end
        end
    end
end

%% display tables
bench_string = strjoin(bench_label, ' ');
var_string = strjoin(variable, ' ');
panel = ['A'; 'B'];

for i=1:size(sample_sel,1)
    disp('------------------------------------------');
    disp(['           TABLE ', num2str(2+i+(loss_type==2)*13),': Forecast Evaluation Tests.']);
    disp('------------------------------------------');

    disp( '------------------------  Sample: '+ string(sample_sel(i,1))+ ' - ' + string(sample_sel(i,2)))

    for d=1:2
        disp(['------------     Panel ',  panel(d)]);

        for j=1:length(variable)
        format bank;
        printmat_LC([squeeze(DM_WCE(1,:,i,j,d)) squeeze(DM_WCE(2,:,i,j,d)); ...
            squeeze(DM_WPE(1,:,i,j,d)) squeeze(DM_WPE(2,:,i,j,d))],...
            variable{j}, 'WCE WPE', ['h=12...', bench_string,' h=24...',bench_string])
        end
    end
end


disp('------------------------------------------');
disp(['           TABLE ',num2str(6 +(loss_type==2)*5),': Average Scores.']);
disp('------------------------------------------');

for i=1:size(sample_sel,1)
    disp(' ------------------------  Sample: '+ string(sample_sel(i,1))+ ' - '+ string(sample_sel(i,2)))
    for j=1:length(variable)
        format bank;
        printmat_LC(tab_6(:,:,i,j),variable{j}, '1Y 2Y', ['SPF ', bench_string],6)
    end
end


for j=1:length(variable)
    disp('------------------------------------------');
    disp(['           TABLE ', num2str(6+j+(loss_type==2)*5) ': Summary Statistics of d_t.']);
    disp('------------------------------------------');
    for d=1:2
        disp(append('------------     Panel ' , panel(d), ': Sample: ' ,...
            string(sample_sel(1,1)) , ' - ' , string(sample_sel(1,2))));
        format bank;
        printmat_LC([squeeze(tab_7(:,1,:,1,j,d)) squeeze(tab_7(:,2,:,1,j,d))],...
            variable{j}, 'Mean std ar1 ar2 ar3 ar4 ar8',...
            ['h=12...', bench_string,' h=24...',bench_string])
    end
end


disp('------------------------------------------');
disp(['           TABLE ',num2str(10+(loss_type==2)*5) ': Estimated Forecast Combination Weights.']);
disp('------------------------------------------');

lambda = max(lambda, 0); %min weight is zero
lambda = min(lambda, 1); %max weight is one

for i=1:size(sample_sel,1)
    disp(' ------------------------  Sample: '+ string(sample_sel(i,1))+ ' - '+ string(sample_sel(i,2)))
    for h=1:2
        format bank;
        printmat_LC(squeeze(lambda(i,:,:,h)), ['h=', num2str(h*12)], var_string, bench_string)
    end
end